// Background script for snARASM (Production 2.0)
console.log('snARASM: Background loaded');

// --- PRODUCTION URL ---
//const API_URL = "https://snarasm.org/api/report/submit";
//staging URL
const API_URL = "https://staging.snarasm.org/api/report/submit";
const SOURCE_ID = "extension-direct";

chrome.runtime.onMessage.addListener((request, sender, sendResponse) => {
    // 1. Auth Flow (Silent -> Interactive)
    chrome.identity.getAuthToken({ interactive: false }, (token) => {
        if (chrome.runtime.lastError || !token) {
            // Retry with popup if silent fails
            chrome.identity.getAuthToken({ interactive: true }, (newToken) => {
                if (newToken) handleRequest(request, newToken, sendResponse);
                else sendResponse({ status: 'error', message: 'Authentication failed.' });
            });
        } else {
            handleRequest(request, token, sendResponse);
        }
    });
    return true; // Keep port open
});

async function handleRequest(request, token, sendResponse) {
    if (request.action === 'reportSpamBatch') {
        const count = await processIds(request.messageIds, token);
        sendResponse({ status: 'success', count: count });
    }
}

// --- CORE LOGIC ---

async function processIds(ids, token) {
    let count = 0;
    const profile = await getProfile(token);
    
    for (let id of ids) {
        // 1. Clean ID: Strip Gmail prefixes (e.g., "thread-f:")
        if (id.includes(':')) {
            id = id.split(':').pop();
        }

        // 2. Clean ID: Convert Decimal to Hex (Gmail API requirement)
        if (/^\d+$/.test(id)) {
            try {
                id = BigInt(id).toString(16);
            } catch (e) {
                console.warn("ID Hex conversion failed, using original");
            }
        }
        
        console.log(`Processing ID: ${id}`);
        
        try {
            // 3. Try fetching as Single Message
            let rawData = await gmailGetRaw(id, token);
            
            if (rawData) {
                await reportAndTrash(id, rawData, profile.email, token);
                count++;
            } else {
                // 4. Fallback: Check if it is a Thread ID
                console.log(`ID ${id} not found as message. Checking as Thread...`);
                const thread = await gmailGetThread(id, token);
                
                if (thread && thread.messages) {
                    // Process all messages in this thread
                    for (const msg of thread.messages) {
                        rawData = await gmailGetRaw(msg.id, token);
                        if (rawData) {
                            await reportAndTrash(msg.id, rawData, profile.email, token);
                            count++;
                        }
                    }
                }
            }
        } catch (e) {
            console.error(`Error processing ${id}`, e);
        }
    }
    return count;
}

async function reportAndTrash(id, rawData, reporter, token) {
    const decoded = decodeBase64Url(rawData);
    const parts = splitContent(decoded);
    
    // Upload to API
    await sendToApi(parts.headers, parts.body, reporter);
    
    // Delete from Gmail
    await gmailTrash(id, token);
    console.log(`Reported & Trashed: ${id}`);
}

// --- HELPERS ---

async function sendToApi(headers, body, reporter) {
    const formData = new URLSearchParams();
    formData.append('headers', headers);
    if(body) formData.append('body', body);
    formData.append('source', SOURCE_ID);
    formData.append('reporter', reporter || 'unknown');

    const res = await fetch(API_URL, {
        method: 'POST',
        headers: {'Content-Type': 'application/x-www-form-urlencoded'},
        body: formData
    });
    
    if (!res.ok) console.error(`API Error: ${res.status}`);
}

async function gmailGetRaw(id, token) {
    const res = await fetch(`https://gmail.googleapis.com/gmail/v1/users/me/messages/${id}?format=raw`, {
        headers: { Authorization: `Bearer ${token}` }
    });
    if (!res.ok) return null;
    const json = await res.json();
    return json.raw;
}

async function gmailGetThread(id, token) {
    const res = await fetch(`https://gmail.googleapis.com/gmail/v1/users/me/threads/${id}`, {
        headers: { Authorization: `Bearer ${token}` }
    });
    if (!res.ok) return null;
    return await res.json();
}

async function gmailTrash(id, token) {
    await fetch(`https://gmail.googleapis.com/gmail/v1/users/me/messages/${id}/trash`, {
        method: 'POST',
        headers: { Authorization: `Bearer ${token}` }
    });
}

async function getProfile(token) {
    try {
        const r = await fetch('https://www.googleapis.com/oauth2/v1/userinfo?alt=json', {
            headers: { Authorization: `Bearer ${token}` }
        });
        return await r.json();
    } catch { return { email: 'unknown' }; }
}

function decodeBase64Url(input) {
    let str = input.replace(/-/g, '+').replace(/_/g, '/');
    while (str.length % 4) str += '=';
    return atob(str);
}

function splitContent(text) {
    const match = text.match(/\r\n\r\n|\n\n/);
    if (match) {
        return {
            headers: text.substring(0, match.index),
            body: text.substring(match.index + match[0].length)
        };
    }
    return { headers: text, body: "" };
}