// content.js - Aggressive ID Finder
console.log('snARASM: Content script loaded');

let isProcessing = false;

function initializeButton() {
    if (document.getElementById('snarasm-report-button')) return;
    const btn = document.createElement('div');
    btn.id = 'snarasm-report-button';
    btn.style.cssText = 'position:fixed;top:70px;right:20px;z-index:9999;padding:10px 18px;background:#667eea;color:white;border-radius:20px;cursor:pointer;font-weight:500;box-shadow:0 2px 8px rgba(0,0,0,0.2);user-select:none;';
    btn.innerHTML = '🛡️ Report';
    btn.onclick = handleReportClick;
    document.body.appendChild(btn);
    document.addEventListener('click', () => setTimeout(updateButtonText, 200), true);
}

if (document.body) initializeButton();
setInterval(updateButtonText, 1000);

// --- THE CRITICAL FIX ---

function getSelectedMessages() {
    const ids = new Set();
    
    // 1. DETAIL VIEW (URL - 100% Reliable)
    const urlMatch = location.hash.match(/\/([a-f0-9]{16,})$/);
    if (urlMatch) return [urlMatch[1]];

    // 2. LIST VIEW (Aggressive Search)
    const checked = document.querySelectorAll('div[role="checkbox"][aria-checked="true"]');
    
    checked.forEach(box => {
        // Walk up to find the row container (tr or div)
        const row = box.closest('tr') || box.closest('div[role="row"]');
        if (!row) return;

        // METHOD A: Standard Input "t"
        const inputT = row.querySelector('input[name="t"]');
        if (inputT && inputT.value) { ids.add(inputT.value); return; }

        // METHOD B: Data Attributes on Row
        if (row.dataset.messageId) { ids.add(row.dataset.messageId); return; }
        if (row.dataset.legacyMessageId) { ids.add(row.dataset.legacyMessageId); return; }

        // METHOD C: Any Hex ID hidden in the row
        // Gmail often hides the ID in span attributes like data-thread-id
        const anyIdEl = row.querySelector('[data-thread-id], [data-legacy-thread-id]');
        if (anyIdEl) {
            ids.add(anyIdEl.getAttribute('data-thread-id') || anyIdEl.getAttribute('data-legacy-thread-id'));
        }
    });

    return Array.from(ids);
}

function updateButtonText() {
    if (isProcessing) return;
    const count = getSelectedMessages().length;
    const btn = document.getElementById('snarasm-report-button');
    if (btn) btn.innerText = count > 0 ? `Report ${count}` : 'Report';
}

function handleReportClick() {
    if (isProcessing) return;
    const ids = getSelectedMessages();
    
    if (ids.length === 0) {
        // Debugging help:
        console.log("snARASM Debug: No IDs found. URL:", location.hash);
        alert('Could not find Message IDs. Please open the email to report it.');
        return;
    }
    
    console.log('Reporting IDs:', ids);
    setProcessingState();
    
    chrome.runtime.sendMessage({
        action: 'reportSpamBatch',
        messageIds: ids 
    }, handleResponse);
}

function handleResponse(response) {
    if (chrome.runtime.lastError) {
        alert('Error: ' + chrome.runtime.lastError.message);
    } else if (response && response.status === 'success') {
        const btn = document.getElementById('snarasm-report-button');
        if(btn) btn.innerText = `✅ Processed ${response.count}`;
        setTimeout(() => {
            document.querySelector('[aria-label="Refresh"]')?.click() || location.reload();
            resetButtonState();
        }, 1500);
    } else {
        alert('Error: ' + (response ? response.message : 'Unknown'));
    }
    resetButtonState();
}

function setProcessingState() {
    isProcessing = true;
    const btn = document.getElementById('snarasm-report-button');
    if(btn) { btn.style.opacity = '0.7'; btn.innerText = '⏳ Processing...'; }
}

function resetButtonState() {
    isProcessing = false;
    const btn = document.getElementById('snarasm-report-button');
    if(btn) { btn.style.opacity = '1'; btn.innerText = '🛡️ Report'; updateButtonText(); }
}